#! /usr/bin/env python

from __future__ import division
from cctbx.array_family import flex
from phenix.substructure import hyss
from phenix.substructure import anomalous_diffs
from phenix.substructure import resolution_cutoff
from phenix import phenix_info
from iotbx import reflection_file_reader
from iotbx import reflection_file_utils
import iotbx.pdb.xray_structure
import iotbx.cns.xray_structure
from iotbx.shelx import write_ins
import iotbx.phil
from cctbx import xray
import cctbx.xray.observation_types
from cctbx import crystal
from cctbx import sgtbx
from cctbx import uctbx
from cctbx.sgtbx import symbol_confidence
import cctbx.eltbx.xray_scattering
import cctbx.eltbx.tiny_pse
from cctbx import eltbx
from libtbx.utils import user_plus_sys_time, human_readable_time
from libtbx.utils import input_with_prompt
from libtbx.math_utils import iround
from libtbx.str_utils import format_value
from libtbx import runtime_utils
from libtbx import easy_pickle
from libtbx.utils import Sorry, Usage
from libtbx import group_args
from libtbx import Auto
import random
import sys, os

url_hyss_documentation = \
  "http://www.phenix-online.org/download/documentation/cci_apps/hyss/"

master_phil = iotbx.phil.parse("""
  data = None
    .type = path
    .short_caption = Anomalous data
    .style = bold file_type:hkl input_file process_hkl child:fobs:data_label \
      child:space_group:space_group child:unit_cell:unit_cell anom
  data_label = None
    .type = str
    .input_size = 160
    .style = bold renderer:draw_fobs_label_widget
  n_sites = None
    .type = int
    .short_caption = Number of sites
    .style = spinner bold
  scattering_type = None
    .type = str
    .input_size = 80
    .style = bold renderer:draw_element_interface
  wavelength = None
    .type = float
  space_group = None
    .type = space_group
  unit_cell = None
    .type = unit_cell
  symmetry = None
    .type = path
    .help = File with crystal symmetry
  resolution = None
    .type = float
    .short_caption = High resolution
  low_resolution = None
    .type = float
    .short_caption = Low resolution
    .help = "Low-resolution limit. Note: if set to None, anomalous differences"
            "will be truncated at resolution of 15 A"
  search = *fast full
    .type = choice(multi=False)
    .short_caption = Search mode
  root = None
    .type = str
    .short_caption = Base output name
    .help = Root for output file names
  output_dir = None
    .type = path
    .short_caption = Output directory
    .style = output_dir
  strategy = *Auto simple brute_force
    .type = choice(multi=False)
    .short_caption = Strategy
    .help = "Strategy.   "
            "Setting strategy=Auto and rescore=Auto will run automatic procedure,"
            "varying resolution and scoring method."
            "Setting strategy=Auto and rescore=phaser-complete will run"
            "phaser completion at varying resolutions."
            "Setting strategy=simple and rescore=correlation will do a single"
            "run of correlation-based scoring at one resolution."
            "Setting strategy=brute_force will use brute-force completion approach."
            "If you set strategy=simple or brute_force then you will "
            "need to set rescore."
  rescore = *Auto correlation phaser-refine phaser-complete
    .type = choice(multi=False)
    .caption = Auto Correlation_coefficient Phaser_refinement Phaser_substructure_completion 
    .short_caption = "Rescoring "
    .help = "Rescore function."
	    "Correlation_coefficient Phaser_refinement and"
            "Phaser_substructure_completion are choices for single-rescoring"
            "procedures."
            "Automatic procedure for substructure determination"
            "assumes: sad=True, wavelength=peak, "
            "variable resolution range if not set.  "
            "Tries first direct methods then"
            "phaser-completion based on Patterson peaks, rescoring "
            "everything at best resolution.  Terminates when "
            "n_top_models_to_compare top-scoring"
            "solutions from different Patterson seeds match."
    .input_size = 200
  llgc_sigma = 5.
    .type = float
    .help = "Log-likelihood gradient map sigma cutoff"
            "Normal value is 5."
            "If your anomalous signal is very low"
            "you might want to try lowering this parameter to as"
            "low as 2."
    .short_caption = Phaser LLG map sigma cutoff
  nproc = None
    .type = int
    .short_caption = Number of processors
    .help = "Number of processors.  NOTE: on Windows nproc=1 always"
    .style = renderer:draw_nproc_widget
  phaser_resolution = None
    .type = float
    .short_caption = High resolution for phaser rescoring and extrapolation
    .help = "Resolution for phaser rescoring and extrapolation."
            "If not set, same as overall resolution."
  phaser_low_resolution = None
    .type = float
    .short_caption = Low resolution for phaser rescoring and extrapolation
    .help = "Low resolution for phaser rescoring and extrapolation."
            "If not set all low-resolution data will be used."
  rms_cutoff = 3.5
    .type = float
    .help = "Anomalous differences larger than rms_cutoff times the rms"
            "will be ignored"
    .short_caption = Outlier sigma cutoff 
  sigma_cutoff = 1
    .type = float
    .help = "Anomalous differences smaller than sima_cutoff times sigma"
            "will be ignored"
    .short_caption = Small anomalous difference cutoff 
  verbose = False
    .type = bool
    .short_caption = Verbose output
  pdb_only = False
    .type = bool
    .help = Suppress all output except for pdb file output
  cluster_termination = False
    .type = bool
    .help = "Terminate by looking for bimodal distribution in phaser"
            "rescoring. Does not apply to correlation rescoring"
    .style = hidden
  max_view = 10
    .type = int
    .help = Number of solutions to show when run in sub-processes
    .style = hidden
  minimum_reflections_for_phaser = 300
    .type = int
    .help = Minimum reflections to run phaser rescoring
    .style = hidden

  # put in params for automatic run
  auto_control {
  lowest_high_resolution_to_try = None
    .type = float
    .short_caption = Lowest resolution high resolution limit in automatic mode
  highest_high_resolution_to_try = None
    .type = float
    .short_caption = Highest resolution high resolution limit in automatic mode
  default_high_resolution_to_try = None
    .type = float
    .short_caption = First high resolution limit to try in automatic mode
  direct_methods = True
    .type = bool
    .short_caption = Direct methods
    .help = "Run direct methods in automatic procedure"
  complete_direct_methods = False
    .type = bool
    .short_caption = Complete direct methods
    .help = "Run Phaser completion on direct methods solutions in "
            "automatic procedure"
  phaser_completion = True
    .type = bool
    .short_caption = Phaser completion
    .help = "Run Phaser completion on Patterson seeds in "
            "automatic procedure"
  try_multiple_resolutions = True
    .type = bool
    .short_caption = Try multiple resolutions if no solution found 
  try_full_resolution = False
    .type = bool
    .short_caption = Try full resolution search if all else fails
  max_multiple = 5
    .type = int
    .short_caption = Max multiple of seeds to try
  max_groups= 5
    .type = int
    .short_caption = Max number of groups of subprocesses 
  seeds_to_use = None
    .type = int
    .short_caption = Seeds to use
  starting_seed = None
    .type = int
    .short_caption = Starting seed
  default_seeds_to_use = 20
    .type = int
    .short_caption = Default value for seeds_to_use
  solutions_to_save = 20
    .type = int
    .help = "If you are running multiprocessing it may be necessary to make"
            "solutions_to_save small (e.g. 10 or 20 rather than 1000)"
    .short_caption = How many top solutions to carry along
  max_tries = None
    .type = int
    .short_caption = Max tries
    .style = hidden
    .help = Used internally by automated procedures
  write_output_model = True
    .type = bool
    .short_caption = write output file
  }
  brute_force_completion {
    n_top_patt = 10
      .type = int
      .short_caption = Patterson solutions for exhaustive completion
      .help = "n_top_patt top-scoring Patterson solutions will be used as"
              "seeds in exhaustive completion" 
    n_top_llg = 10
      .type = int
      .short_caption = LLG solutions for exhaustive completion
      .help = "n_top_llg LLG map peaks will be used at a time"
              "in exhaustive completion" 
    min_sigma_in_brute_force = 1.
      .type = float
      .short_caption = Minimum LLG sigma in brute force completion
    n_top_iter = 3
      .type = int
      .short_caption = Top groups for iteration of exhaustive completion
      .help = "n_top_iter top groups will be carried forward"
              "in exhaustive completion" 
    n_llg_add_at_once = 1
      .type = int
      .short_caption = Number of LLG peaks to add each round
      .help = "All possible groups of n_llg_add_at_once of the n_top_llg LLG peaks will "
              "be added to each seed in each round."
              "As the addition is exhaustive, n_llg_add_at_once more than 3 may take a"
              "really long time."
    n_iter_max = 3
      .type = int
      .short_caption = Iterations of exhaustive completion
      .help = "Number of iterations of exhaustive completion"

    include_base_scatterers = True
      .type = bool
      .short_caption = Include fragments each cycle
      .help = Include fragments used to run LLG map in new fragments
    score_additions_with_completion = True
      .type = bool
      .short_caption = Score additions with completion
      .help = "Score compound fragments (patterson + additional sites)"
              "using phaser-completion (llgc_sigma not set to 1000)"
    select_matching = None
      .type = int
      .short_caption = Select solutions matching at least N sites in comparison model
      .help = "Not normally used.  Choose n_llg_add_at_once+2 (patterson+add)"
      .style = hidden 
  }
  termination {
  matches_must_not_share_patterson_vector = True
   .type = bool
   .short_caption = One match per Patterson Vector
  n_top_models_to_compare = None
   .help = "Number of top models that must share sites to terminate (set"
           "automatically."
   .type = int
   .short_caption = Number of top models that must share sites to terminate 
  }
  parameter_estimation {
  solvent_content = 0.55
    .type = float
    .help = Solvent content (default: 0.55)
    .expert_level = 2
  molecular_weight = None
    .type = int
    .expert_level = 2
  }
  search_control {
  random_seed = None
    .type = int
    .help = Seed for random number generator
  site_min_distance = 3.5
    .type = float
    .help = Minimum distance between substructure sites (default: 3.5)
  site_min_distance_sym_equiv = None
    .type = float
    .help = Minimum distance between symmetrically-equivalent \
            substructure sites (overrides --site_min_distance)
  site_min_cross_distance = None
    .type = float
    .help = Minimum distance between substructure sites not \
            related by symmetry (overrides --site_min_distance)
  skip_consensus = False
    .type = bool
    .short_caption = Skip consensus
  }
  direct_methods{
  real_space_squaring = False
    .type = bool
    .help = Use real space squaring (as opposed to the tangent formula)
  extrapolation = *fast_nv1995 phaser-map
    .type = choice(multi=False)
    .help = Extrapolation function
  }
  fragment_search {
  minimum_fragments_to_consider = 0
    .type = int
    .short_caption = Minimum fragments to consider
    .help = Minimum fragments per patterson vector
  n_patterson_vectors = None
    .type = int
    .short_caption = Patterson vectors to consider
    .help = Patterson vectors to consider
  score_initial_solutions = False
    .type = bool
    .short_caption = Score initial solutions
    .help = Score 2-site Patterson solutions and sort them
  input_emma_model_list = None
    .type = path
    .help = "Input model or models. "
            "Each model group in a file is treated separately."
  input_add_model = None
    .type = path
    .short_caption = Input atoms to add to input_model
    .help = Input atoms to add to input_model
    .style = file_type:pdb
  keep_seed_sites = False
    .type = bool
    .short_caption = Keep seed sites
    .help = Keep seed sites (never omit them)
  n_shake= 0
    .type = int
    .short_caption = Number of random variants of each fragment
    .help = Number of random variants of each fragment
  rms_shake= 1.
    .type = float
    .short_caption = RMS for random variants
    .help = RMS for random variants
  dump_all_fragments = False
    .type = bool
    .help = Dump fragments to fragment_x.pdb. Requires score_initial_solutions=True
  score_only = False
    .type = bool
    .short_caption = Just score input models
    .help = Just score input models
  dump_all_models = False
    .type = bool
    .help = Dump models to dump_xxx_yy.pdb xxx=patterson yy=trans
  }
  comparison_files {
  comparison_emma_model = None
    .type = path
    .short_caption = Comparison model or models
    .help = Comparison model or models
  comparison_emma_model_tolerance = 1.5
    .type = float
    .short_caption = Comparison model tolerance
    .help = Comparison model tolerance
  }
  include scope libtbx.phil.interface.tracking_params
  chunk
    .expert_level = 3
  {
    n = 1
      .type = int
      .expert_level = 3
    i = 0
      .type = int
      .expert_level = 3
  }
""", process_includes=True)
master_params = master_phil

class process_command_line (iotbx.phil.process_command_line_with_files) :
  def process_other (self, arg) :
    if len(arg)>2: 
      return False
    try :
      scattering_type = process_scattering_type(arg)
    except Exception :
      return False
    else :
      return iotbx.phil.parse("scattering_type='%s'" % arg)

def get_emma_model_list_from_pdb(file_name):
    from phenix.autosol.get_pdb_inp import get_pdb_inp
    from iotbx.command_line.emma import get_emma_model_from_pdb
    emma_model_list=[]
    pdb_inp=get_pdb_inp(file_name=file_name)
    hierarchy=pdb_inp.construct_hierarchy()
    for model in hierarchy.models():
      lines=[]
      for atom in model.atoms():
        lines.append(atom.format_atom_record())
      emma_model_list.append(get_emma_model_from_pdb(pdb_records=lines,
        crystal_symmetry=pdb_inp.crystal_symmetry()))
 
    return emma_model_list

def run(args,input_arrays=None,out=sys.stdout,
    min_anom_diffs=100):
  timer = user_plus_sys_time()
  if args==["--help"]: args=[]
  print >>out,"Command line arguments:",
  for arg in args: print >>out,arg,
  print >>out
  print >>out
  command_line = process_command_line(
    args=args,
    master_phil=master_phil,
    reflection_file_def="data",
    pdb_file_def="symmetry",
    integer_def="n_sites",
    float_def="wavelength")
  params = command_line.work.extract()
  master_phil.format(python_object=params).show(out=out)

  if (len(args) == 0) or ("--help" in args) :
    raise Usage("""\
phenix.hyss reflection_file n_sites element_symbol [options]
See also:
  %s

Example: phenix.hyss w1.sca 66 Se
""" % url_hyss_documentation)

  validate_params_basic(params,input_arrays=input_arrays)

  phenix_info.show_banner(f=out)
  print >>out,"                       HySS: Hybrid Substructure Search"
  print >>out
  phenix_info.show_developers(f=out)
  phenix_info.show_copyright(f=out)
  phenix_info.show_third_party_summary(f=out)

  if (params.search_control.random_seed is not None):
    random.seed(params.search_control.random_seed)

  n_sites = params.n_sites
  scatterer, general_positions_only = process_scattering_type(
    params.scattering_type)
  if params.data:
    reflection_file_name = str(params.data)
  else:
    reflection_file_name = 'miller_arrays'
  if not input_arrays:
    print >>out,"Reading reflection file:", reflection_file_name
    reflection_file = reflection_file_reader.any_reflection_file(
      reflection_file_name)
    print >>out
    symmetry = None
    if (params.symmetry is not None) :
      from iotbx import crystal_symmetry_from_any
      symmetry = crystal_symmetry_from_any.extract_from(
        file_name=params.symmetry)
    command_line_symmetry = prompt_for_symmetry_if_necessary(
      command_line_symmetry=symmetry,
      reflection_file=reflection_file,
      unit_cell=params.unit_cell,
      space_group_info=params.space_group,out=out)

    # Get data as input 
    input_arrays = reflection_file.as_miller_arrays(
      crystal_symmetry=command_line_symmetry,
      force_symmetry=True)
  if (params.data_label is not None):
    print >>out,"\n   SELECTING ANOMALOUS DATA BASED ON LABEL OF %s\n" %(params.data_label)
    ( original, anom_diffs ) = match_data_label(
      input_arrays=input_arrays,
      data_label=params.data_label,
      rms_cutoff=params.rms_cutoff,
      sigma_cutoff=params.sigma_cutoff,
      out=out)
  else:
    print >>out,"\n   SELECTING ANOMALOUS DATA\n"
    ( original, anom_diffs ) = select_array_to_use(
      input_arrays=input_arrays,
      rms_cutoff=params.rms_cutoff,
      sigma_cutoff=params.sigma_cutoff,
      out=out)

  # remove any negative sigmas:  2014-03-17
  if original:
    original=original.select(original.sigmas() > 0)
  stop_now=False
  if anom_diffs.data().size() < min_anom_diffs:
    line="\nNot enough anom diffs (%d) to run...quitting\n" %(
       anom_diffs.data().size())
    stop_now=True
  elif original is None and (
      params.rescore in ['phaser-refine','phaser-complete'] or
     (params.rescore=='Auto' and (
        params.auto_control.phaser_completion or 
        params.auto_control.complete_direct_methods)
     )
    ): 
    line="\nSorry, unable to run HySS with unmerged data and phaser scoring\n"
    stop_now=True

  elif params.rescore == 'Auto' and params.fragment_search.score_only:
    line="Sorry need to set rescoring type to non-Auto for score_only"
    stop_now=True

  elif params.rescore == 'Auto' and params.strategy != 'Auto':
    line="Sorry need to set rescoring type to non-Auto for strategy=%s" %(
       params.strategy) 
    stop_now=True

  if stop_now:
    raise Sorry(line)

  # now original=original data; anom_diffs=anomalous differences. No res cutoffs

  if (params.comparison_files.comparison_emma_model is not None):
    from iotbx.command_line.emma import get_emma_model_from_pdb
    spl=params.comparison_files.comparison_emma_model.replace(","," ").split()
    comparison_emma_model=[]
    for x in spl:
     if x and os.path.isfile(x):
      comparison_emma_model.append(get_emma_model_from_pdb(x))
      print >>out, "\nComparing all solutions to comparison emma model:"
      xray_scatterer=xray.scatterer(scattering_type=scatterer.scattering_type)
      print >>out,comparison_emma_model[-1].as_xray_structure(
         xray_scatterer).as_pdb_file()

  else:
    comparison_emma_model=None

  if (params.fragment_search.input_add_model is not None):
    from iotbx.command_line.emma import get_emma_model_from_pdb
    input_add_model=get_emma_model_from_pdb(
       params.fragment_search.input_add_model)
  else:
    input_add_model=None

  if (params.fragment_search.input_emma_model_list is not None):
    from iotbx.command_line.emma import get_emma_model_from_pdb
    spl=params.fragment_search.input_emma_model_list.replace(","," ").split()
    input_emma_model_list=[]
    for x in spl:
     if x and os.path.isfile(x):
      input_emma_model_list+=get_emma_model_list_from_pdb(x)
    params.auto_control.max_multiple=1  # because we have just the input models
  else:
    input_emma_model_list=None


  if (params.parameter_estimation.molecular_weight is not None):
    molecular_weight = params.parameter_estimation.molecular_weight
    solvent_content = params.parameter_estimation.solvent_content
    v_cell = anom_diffs.unit_cell().volume()
    n_sym = anom_diffs.space_group().order_z()
    n_mol = iround(((1.0-solvent_content)*v_cell)/(molecular_weight*n_sym*.783))
    n_sites *= n_mol
    print >>out,"#---------------------------------------------------------------------------#"
    print >>out,"| Formula for calculating the number of molecules given a molecular weight. |"
    print >>out,"|---------------------------------------------------------------------------|"
    print >>out,"| n_mol = ((1.0-solvent_content)*v_cell)/(molecular_weight*n_sym*.783)      |"
    print >>out,"#---------------------------------------------------------------------------#"
    print >>out,"Number of molecules:", n_mol
    print >>out,"Number of sites:", n_sites
    print >>out,"Values used in calculation:"
    print >>out,"  Solvent content: %.2f" % solvent_content
    print >>out,"  Unit cell volume: %.6g" % v_cell
    print >>out,"  Molecular weight: %.2f" % molecular_weight
    print >>out,"  Number of symmetry operators:", n_sym
    print >>out

  output_prefix = os.path.splitext(os.path.split(reflection_file_name)[-1])[0]
  if (params.chunk.n != 1):
    output_prefix += "_%02d" % params.chunk.i
  if (params.root is not None):
    output_prefix = params.root

  if (not params.pdb_only):
    shelx_hklf = output_prefix + "_anom_diffs.hkl"
    print >>out,"Writing anomalous differences as SHELX HKLF file:", shelx_hklf
    anom_diffs.export_as_shelx_hklf(open(shelx_hklf, "w"))
    print >>out

  substructure_params = hyss.substructure_parameters(
    n_sites=n_sites,
    min_distance=params.search_control.site_min_distance,
    general_positions_only=general_positions_only,
    min_distance_sym_equiv=params.search_control.site_min_distance_sym_equiv,
    min_cross_distance=params.search_control.site_min_cross_distance)

  if (not params.pdb_only):
    shelxd_ins = output_prefix + "_anom_diffs.ins"
    print >>out,"Writing SHELXD ins file:", shelxd_ins
    if (substructure_params.general_positions_only):
      mind_mdeq = None
    else:
      mind_mdeq = -0.1
    write_ins.shelxd(open(shelxd_ins, "w"),
      title=shelxd_ins,
      crystal_symmetry=anom_diffs,
      n_sites=n_sites,
      scattering_type=scatterer.scattering_type,
      d_min=anom_diffs.d_min(),
      mind_mdis=-substructure_params.min_distance,
      mind_mdeq=mind_mdeq)
    print >>out
  if (params.search == "full" or params.strategy == 'brute_force'):
    enable_early_termination = False
  else:
    enable_early_termination = True


  if params.auto_control.phaser_completion or \
        params.auto_control.complete_direct_methods:
    if not params.wavelength and params.scattering_type.lower()=='se': 
      params.wavelength=0.9792 
      print >>out,\
         "Using default wavelength of %8.4f for scattering type '%s'\n" %(
         params.wavelength,params.scattering_type)
    elif not params.wavelength:
      raise Sorry("\nPhaser completion and LLG maps require a wavelength with "+
          "command like 'wavelength=0.9792' \n"+
          "Alternatively you can set 'phaser_completion=False' and \n"+
          "'complete_direct_methods=False' and 'strategy=simple' or Auto")
   
  if params.resolution and not params.phaser_resolution:
    params.phaser_resolution=params.resolution 
  if params.low_resolution and not params.phaser_low_resolution:
    params.phaser_low_resolution=params.low_resolution 

  # set up search parameters
  search_params=hyss.search_parameters(
      i_chunk=params.chunk.i,
      n_chunk=params.chunk.n,
      enable_early_termination=enable_early_termination,
      real_space_squaring=params.direct_methods.real_space_squaring,
      max_tries = params.auto_control.max_tries,
      comparison_emma_model=comparison_emma_model,
      comparison_emma_model_tolerance=\
         params.comparison_files.comparison_emma_model_tolerance,
      skip_consensus=\
         params.search_control.skip_consensus,
      input_add_model=input_add_model,
      score_only=params.fragment_search.score_only,
      input_emma_model_list=input_emma_model_list,
      keep_seed_sites=params.fragment_search.keep_seed_sites,
      verbose=params.verbose,
      cluster_termination=params.cluster_termination,
      max_view=params.max_view,
      minimum_reflections_for_phaser=params.minimum_reflections_for_phaser,
      dump_all_fragments=params.fragment_search.dump_all_fragments,
      dump_all_models=params.fragment_search.dump_all_models,
      n_shake=params.fragment_search.n_shake,
      rms_shake=params.fragment_search.rms_shake,
      score_initial_solutions=params.fragment_search.score_initial_solutions,
      n_patterson_vectors=params.fragment_search.n_patterson_vectors,
      minimum_fragments_to_consider=\
         params.fragment_search.minimum_fragments_to_consider,
      matches_must_not_share_patterson_vector=\
        params.termination.matches_must_not_share_patterson_vector,
      n_top_models_to_compare=params.termination.n_top_models_to_compare,
      )

  print >>out,"Starting the search procedure (%s mode):" %params.search
  print >>out

  if params.strategy in ['Auto','brute_force']:
    automatic=True
    search_result=run_automatic(params=params,search_params=search_params,
       substructure_params=substructure_params,
       scatterer=scatterer,
       original=original,anom_diffs=anom_diffs,out=out)

  else:  # standard run
    automatic=False
    search_result=run_group(params=params,search_params=search_params,
       substructure_params=substructure_params,
       scatterer=scatterer,
       original=original,anom_diffs=anom_diffs,out=out,
       standard_run=True)

  print >>out
  if (search_result and search_result.structures and not params.pdb_only):
    hyss_models_file_name = output_prefix + "_hyss_models.pickle"
    print >>out,"Storing all substructures found:", hyss_models_file_name
    easy_pickle.dump(hyss_models_file_name, search_result.structures)
    print >>out

  pdb_file_name = None
  xyz_file_name = None
  if ( search_result and (not search_result.failed) and \
      search_result.top_matches is not None
      and search_result.top_matches.n_matches() > 0
      and search_result.consensus_model):

    print >>out," --- TOP-SCORING MODEL OVERALL ---\n"
    print >>out,"Score: ",search_result.consensus_score
    if search_result.terminated_early:
      print >>out,"This search terminated early based on matching solutions"
    
    hyss_consensus_model_base_name = output_prefix + "_hyss_consensus_model"

    hyss_consensus_model_file_name = hyss_consensus_model_base_name + ".pickle"
    consensus_model = search_result.consensus_model

    if (not params.pdb_only):
      print >>out,"Storing consensus model:", hyss_consensus_model_file_name
      easy_pickle.dump(hyss_consensus_model_file_name, consensus_model)
      print >>out

    pdb_file_name = hyss_consensus_model_base_name + ".pdb"
    if params.auto_control.write_output_model:
      print >>out,"Writing consensus model as PDB file:", pdb_file_name

    s=consensus_model_as_pdb_string(consensus_model=consensus_model,
      scatterer=scatterer)
    print >>out, "\n%s" %(s)
    if params.auto_control.write_output_model:
      f = open(pdb_file_name, "w")
      f.write(s)
      f.close()
    print >>out

    if (not params.pdb_only):
      sdb_file_name =  hyss_consensus_model_base_name + ".sdb"
      print >>out,"Writing consensus model as CNS SDB file:", sdb_file_name
      sdb_structure = xray.structure(consensus_model)
      for i,sc in enumerate(consensus_model.scatterers()):
        sdb_structure.add_scatterer(sc.customized_copy(
          scattering_type=scatterer.scattering_type.upper()))
      s = sdb_structure.as_cns_sdb_file(
        file=sdb_file_name,
        description="phenix.hyss consensus model")
      f = open(sdb_file_name, "w")
      f.write(s)
      f.close()
      print >>out

    if (not params.pdb_only):
      xyz_file_name =  hyss_consensus_model_base_name + ".xyz"
      print >>out,"Writing consensus model as SOLVE xyz records:", xyz_file_name
      print >>out,"The fractional coordinates may also be useful in other programs."
      f = open(xyz_file_name, "w")
      print >> f, "atomname", scatterer.scattering_type.lower()
      for sc in consensus_model.scatterers():
        print >> f, "xyz %10.6f %10.6f %10.6f" % sc.site
      f.close()
      print >>out

    print >>out, "If HySS was helpful please cite:"
    print >>out, "  Acta Cryst. 2003, D59, 1966-1973"
    print >>out, "  http://cci.lbl.gov/publications/download/ba5048_reprint.pdf"
    print >>out, "See also:"
    print >>out, " ", url_hyss_documentation
    print >>out, "For help enter:"
    print >>out, "  phenix.hyss --help"
    print >>out
  phenix_info.show_phenix_email_addresses(f=out)
  print >>out

  total_time = timer.delta()
  try: python_ticker = sys.gettickeraccumulation()
  except AttributeError: pass
  else:
    print >>out,"Time per interpreted Python bytecode instruction:",
    print >>out,"%.3f micro seconds" % (total_time / python_ticker * 1.e6)
  print >>out,"Total CPU time: %.2f %s" % human_readable_time(total_time)

  if search_result:
    cc, n_sites = search_result.get_consensus_cc_and_site_count()
  else:
    cc, n_sites= 0.0, 0
  if (pdb_file_name is not None) :
    pdb_file_name = os.path.abspath(pdb_file_name)
  if (xyz_file_name is not None) :
    xyz_file_name = os.path.abspath(xyz_file_name)

  if automatic and not params.auto_control.write_output_model:
    # no files written, just return cc,ha_model,occ 
    ha_model=[]
    occupancies=[]
    if search_result:
      for atom in consensus_model.scatterers():
        occupancies.append(atom.occupancy)
        coords=[]
        for xyz in atom.site:
          coords.append(xyz)
        ha_model.append(coords)
      print >>out, "HA MODEL: "+str(ha_model)
      return  [cc,ha_model,occupancies],search_result.terminated_early
    else:
      return [0.0,[],[]],False

  else: # usual
    return group_args(
      pdb_file=pdb_file_name,
      xyz_file=xyz_file_name,
      cc=cc,
      n_sites=n_sites)

def consensus_model_as_pdb_string(consensus_model=None,scatterer=None):
    pdb_structure = xray.structure(consensus_model)
    for i,sc in enumerate(consensus_model.scatterers()):
      pdb_structure.add_scatterer(sc.customized_copy(
        label="%2s"%scatterer.scattering_type,
        scattering_type=scatterer.scattering_type))
    s = pdb_structure.as_pdb_file(
      remark="phenix.hyss consensus model",
      remarks=[],
      resname="SUB")
    return s

def max(i,j):
  if i> j: return i
  return j

def min(i,j):
  if i < j: return i
  return j

def get_lowest_high_resolution(params=None,crystal_symmetry=None,
    n_sites=None,full_resolution=None,out=sys.stdout,
    default_lowres=9):

  # take 5 A or high_res + 2 for small cell; 10 A or high_res+4 for big one
  au_volume=crystal_symmetry.unit_cell().volume()/float(
      crystal_symmetry.space_group().n_equivalent_positions())
  volume_per_site=au_volume/float(n_sites)
  half_dist_between_sites=float(int(0.5+0.5*volume_per_site**0.333))
  print >>out, "Volume of asymmetric unit: %10.1f A**3" %(au_volume)
  print >>out, "Volume per site : %10.1f A**3" %(volume_per_site)
  print >>out, "Half distance between sites : %7.1f A**3" %(
     half_dist_between_sites)
  print >>out, "High-resolution: %7.1f" %(full_resolution)
  # bracket between 1/2 of half-distance and high_resolution at most
  
  if params.auto_control.lowest_high_resolution_to_try:
    lowest_high_resolution=params.auto_control.lowest_high_resolution_to_try
    print >>out,"Using input lowest_high_resolution of %7.1f \n" %(
      lowest_high_resolution)
  else:
    target=max(full_resolution+2.,6.0) # always at least this
    target=max(target,min(default_lowres,0.5*half_dist_between_sites))
    lowest_high_resolution=target
    print >>out,"Using lowest_high_resolution of %7.1f \n" %(
      lowest_high_resolution)

  return lowest_high_resolution

def similar_resolution(r,resolution_series,delta=0.3):
  for res in resolution_series:
    if res > r-delta  and res < r+delta:
      return True
  return False

def add_to_resolution_series(params=None,original_d_min=None,
   lowest_high_resolution=None,resolution_series=[],out=sys.stdout):

    if params.auto_control.default_high_resolution_to_try:
      print >>out,"First high_resolution set by user to %7.1f" %(
         params.auto_control.default_high_resolution_to_try)
      resolution_series.append(
       params.auto_control.default_high_resolution_to_try)

    standard_default=float("%7.1f" %(original_d_min+0.5))
    if params.auto_control.highest_high_resolution_to_try:
      default=params.auto_control.highest_high_resolution_to_try
      print >>out,"Highest high_resolution set by user to %7.1f" %(
         default)
      resolution_series.append(default)
    elif params.auto_control.default_high_resolution_to_try:
      default=params.auto_control.default_high_resolution_to_try-1.0
      default=float("%7.1f" %(max(original_d_min+0.5,default)))
      print >>out,"Highest high_resolution set to %7.1f" %(
         default)
      if not similar_resolution(default,resolution_series):
        resolution_series.append(default)
    else:
      default=standard_default
      print >>out,"Highest high_resolution set to %7.1f" %( default)
      if not similar_resolution(default,resolution_series):
        resolution_series.append(default)

    # start at default + 0.5 and go up in 1 A increments
    # default =3.1: 4 5 6; 3.9:  5 6; 3.8: 3.8 4 5 6
    highest_high_resolution=standard_default

    if params.auto_control.try_multiple_resolutions:
      delta=1
      n=0
      x=highest_high_resolution
      while x < lowest_high_resolution+0.1:
        if not similar_resolution(x,resolution_series):
          resolution_series.append(x)
        x+=delta
        n+=1
        if n>=2:
          n=0
          delta+=delta
    return resolution_series

def get_search_params_series(params=None,original_d_min=None,
  substructure_params=None,search_params=None,
  crystal_symmetry=None,out=sys.stdout):

  # set up a series of search params that we will work through until
  #  we get a good answer or run out of things to try
  # NOTE: return resolution cutoff as search_params.high_resolution

  # identify default search parameters that we are going to vary
  from copy import deepcopy
  search_params_default=deepcopy(search_params) 
  search_params_default.compute_derived(n_sites=substructure_params.n_sites)

  patterson_vectors_series=[]
  seeds_to_use_series=[]
  if params.auto_control.seeds_to_use:
    default_seeds_to_use=params.auto_control.seeds_to_use
  else:
    default_seeds_to_use=params.auto_control.default_seeds_to_use
  if params.fragment_search.n_shake and params.fragment_search.n_shake>1:
      default_seeds_to_use=default_seeds_to_use*params.fragment_search.n_shake
    
  if params.strategy != "brute_force":
    print >>out,"Base number of seeds: %d" %( default_seeds_to_use)
  
  if params.fragment_search.n_patterson_vectors:
    default_n_patterson_vectors=params.fragment_search.n_patterson_vectors
  else:
    default_n_patterson_vectors=search_params_default.n_patterson_vectors
  if params.auto_control.max_multiple <= 1:
    multiple_list=[1]
  else:
    multiple_list=[1,params.auto_control.max_multiple]
  max_multiple=multiple_list[-1]
   
  print >>out,"Multiples of seeds to try: ",multiple_list
  for multiple in multiple_list:
    patterson_vectors_series.append(default_n_patterson_vectors*max_multiple)
    seeds_to_use_series.append(max(params.nproc,default_seeds_to_use*multiple))
  print >>out
  resolution_series=[]
  if params.resolution:
    resolution_series.append(params.resolution)
  else:
    full_resolution=float("%7.1f" %(original_d_min))
    lowest_high_resolution=get_lowest_high_resolution(params=params,
      crystal_symmetry=crystal_symmetry,
      full_resolution=full_resolution,
      n_sites=substructure_params.n_sites,
      out=out)

    resolution_series=add_to_resolution_series(
      resolution_series=resolution_series,
      lowest_high_resolution=lowest_high_resolution,
      params=params,original_d_min=original_d_min,out=out)

    if params.auto_control.try_full_resolution and \
        not full_resolution in resolution_series:
        resolution_series.append(full_resolution)
  print >>out,"High-resolution limits to be tried:",
  for x in resolution_series: print >>out," %6.1f A" %(x),
  print >>out

  if params.search != "full" and params.strategy != "brute_force":
    # number of top matches required to terminate 
    print >>out, "Note: stopping early if top %d solutions match\n" %(
      search_params_default.n_top_models_to_compare)
 
  # now set up parameters in order we want to run them.
  search_params_list=[]
  starting_seed=0
  for n_patterson_vectors,seeds_to_use \
       in zip(
       patterson_vectors_series,
       seeds_to_use_series,
       ):
    for resolution in resolution_series: 
      search_params_local=deepcopy(search_params) 
      search_params_local.high_resolution_cutoff=resolution
      search_params_local.n_patterson_vectors=n_patterson_vectors

      search_params_local.seeds_to_use=seeds_to_use
      search_params_local.starting_seed=starting_seed

      search_params_list.append(search_params_local)
    # at end increment starting seed
    starting_seed+=seeds_to_use
  
  return search_params_list
      

def run_automatic(params=None,search_params=None,
  substructure_params=None, scatterer=None,
  original=None,anom_diffs=None,out=sys.stdout):
  
  # for automatic run a set of searches and pick the best
  # if brute_force=True then do brute force completion

  print >>out," ---  RUNNING IN AUTOMATIC MODE --- \n"
 
  # set any things that need to be the same for all runs
  if not params.nproc: params.nproc=1

  fix_patterson_vectors=False
  if params.strategy=='brute_force':
    print >>out,"RUNNING BRUTE-FORCE SEARCH"
    params.auto_control.direct_methods=False
    params.auto_control.complete_direct_methods=False
    params.auto_control.phaser_completion=False

  from copy import deepcopy
  orig_search_params=deepcopy(search_params) 
  orig_params=deepcopy(params)
  orig_substructure_params=deepcopy(substructure_params)
  done=False
  original_d_min=anom_diffs.d_min()

  # NOTE: need to set params.rescore for each call  2014-05-14

  search_params_series=get_search_params_series(params=orig_params,
     search_params=orig_search_params,
     substructure_params=orig_substructure_params,
     crystal_symmetry=anom_diffs.crystal_symmetry(),
     original_d_min=original_d_min,out=out)

  # NOTE: best_search_result is a hyss.search result
  #       previous_search_result's are hyss.subproccess_search_result objects.
  patterson_fragment_dict={}
  best_search_result_dict={}
  previous_search_result_complete_direct_methods_dict={} 
  previous_search_result_direct_methods_dict={}
  previous_search_result_phaser_completion_dict={}

  overall_best_search_result=None
  done=False
  for working_search_params in search_params_series:
    if done: break  # Need done and terminated_early for best
    resolution=working_search_params.high_resolution_cutoff
    print >>out,"\n --NEW SET OF SEARCH PARAMETERS --\n"
    print >>out,"Resolution: %7.2f A\n" %(resolution)
    if resolution in best_search_result_dict.keys():
      best_search_result=best_search_result_dict[resolution]
      previous_search_result_direct_methods=\
         previous_search_result_direct_methods_dict[resolution]
      previous_search_result_complete_direct_methods=\
        previous_search_result_complete_direct_methods_dict[resolution]
      previous_search_result_phaser_completion=\
        previous_search_result_phaser_completion_dict[resolution]
      if best_search_result:
        print >>out,"Starting from best search result "+\
         "with %d structures at resolution of %6.1f A "%(
         len(best_search_result.structures),resolution)
      if previous_search_result_direct_methods:
        print >>out,"NOTE: direct_methods structures: %d" %(
           len(previous_search_result_direct_methods.structures))
      if previous_search_result_complete_direct_methods:
        print >>out,"NOTE: complete_direct_methods structures: %d" %(
           len(previous_search_result_complete_direct_methods.structures))
      if previous_search_result_phaser_completion:
        print >>out,"NOTE: phaser_completion structures: %d" %(
           len(previous_search_result_phaser_completion.structures))
    else:
      previous_search_result_direct_methods=None
      previous_search_result_complete_direct_methods=None
      previous_search_result_phaser_completion=None
      best_search_result=None
      print >>out,"No previous search results at this resolution"
    if resolution in patterson_fragment_dict.keys():
      patterson_fragment_list=patterson_fragment_dict[resolution]
    else:
      # Get patterson vectors   2014-05-14 if input_emma_model_list is set use them
      patterson_fragment_list,cb_op_niggli=run_standard( # get patterson seeds
        high_resolution_cutoff=working_search_params.high_resolution_cutoff,
        patterson_seeds_only=True,
        orig_search_params=working_search_params,
        orig_params=orig_params,
        orig_substructure_params=orig_substructure_params,
        allow_shake=True,
        scatterer=scatterer,original=original,anom_diffs=anom_diffs,out=out)
      patterson_fragment_dict[resolution]=patterson_fragment_list
      # if working_search_params had input_emma_model_list, remove it now (it was used
      #  to replace patterson vectors)
      # Also turn of shake if it was present
      fix_patterson_vectors=True
    if fix_patterson_vectors:
      working_search_params.input_emma_model_list=None
      working_search_params.n_shake=None
      

    if params.strategy=='brute_force':
      if search_params.input_emma_model_list:
        seed_text="INPUT FRAGMENTS"
      else:
        seed_text="PATTERSON"
      best_search_result,done,previous_search_result_phaser_completion=\
       run_brute_force(
          cb_op_niggli=cb_op_niggli,
          seed_text=seed_text,
          high_resolution_cutoff=resolution,
          fragment_list=patterson_fragment_list,
          best_search_result=best_search_result,
          previous_search_result=previous_search_result_phaser_completion,
          orig_search_params=working_search_params,
          orig_params=orig_params,
          orig_substructure_params=orig_substructure_params,
          scatterer=scatterer,original=original,anom_diffs=anom_diffs,out=out)

    if params.auto_control.direct_methods:
      # run direct methods from patt
      best_search_result,done,previous_search_result_direct_methods=\
         run_standard(
        high_resolution_cutoff=resolution,
        starting_seed=working_search_params.starting_seed,
        seeds_to_use=working_search_params.seeds_to_use,
        fragment_list=patterson_fragment_list,
        best_search_result=best_search_result,
        previous_search_result=previous_search_result_direct_methods,
        orig_search_params=working_search_params,
        orig_params=orig_params,
        orig_substructure_params=orig_substructure_params,
        scatterer=scatterer,original=original,anom_diffs=anom_diffs,out=out)
      if not done and params.auto_control.complete_direct_methods:
        direct_methods_search_result=hyss.subprocess_search_result(
          best_search_result,
          max_structures=params.auto_control.solutions_to_save)
        direct_methods_fragment_list=get_fragment_list(
          search_result=direct_methods_search_result,
          cb_op_niggli=cb_op_niggli)

        best_search_result,done,previous_search_result_complete_direct_methods=\
            run_phaser_completion_on_existing_solutions(
          cb_op_niggli=cb_op_niggli,
          seed_text="DIRECT METHODS",
          high_resolution_cutoff=resolution,
          starting_seed=0,
          seeds_to_use=working_search_params.seeds_to_use,
          best_search_result=best_search_result,
          previous_search_result=previous_search_result_complete_direct_methods,
          fragment_list=direct_methods_fragment_list,
          orig_search_params=working_search_params,
          orig_params=orig_params,
          orig_substructure_params=orig_substructure_params,
          scatterer=scatterer,original=original,anom_diffs=anom_diffs,out=out)

    if not done and params.auto_control.phaser_completion:
      # run phaser completion on patterson seeds
 
      best_search_result,done,previous_search_result_phaser_completion=\
       run_phaser_completion_on_existing_solutions(
          cb_op_niggli=cb_op_niggli,
          seed_text="PATTERSON",
          high_resolution_cutoff=resolution,
          starting_seed=working_search_params.starting_seed,
          seeds_to_use=working_search_params.seeds_to_use,
          fragment_list=patterson_fragment_list,
          best_search_result=best_search_result,
          previous_search_result=previous_search_result_phaser_completion,
          orig_search_params=working_search_params,
          orig_params=orig_params,
          orig_substructure_params=orig_substructure_params,
          scatterer=scatterer,original=original,anom_diffs=anom_diffs,out=out)

    # save current best result for this resolution...
    best_search_result_dict[resolution]=best_search_result
    previous_search_result_direct_methods_dict[resolution]=\
       previous_search_result_direct_methods
    previous_search_result_complete_direct_methods_dict[resolution]=\
       previous_search_result_complete_direct_methods
    previous_search_result_phaser_completion_dict[resolution]=\
       previous_search_result_phaser_completion
   
    #and save overall best so far:
    print >>out,"\n--GETTING BEST OVERALL SEARCH RESULT SO FAR--"
    overall_best_search_result=get_best_search_result(
        overall_best_search_result,search_result=best_search_result,
        scatterer=scatterer,params=params,out=out)
    if not overall_best_search_result or \
       not overall_best_search_result.terminated_early:
      done=False # keep trying 2014-02-05
  return overall_best_search_result

def select_good(fragments,search_result,matching=None,out=sys.stdout):
  if not matching or \
     not search_result or not search_result.search_params.comparison_emma_model:
    return fragments

  new_fragments=[]
  for f in fragments:
    n_match=search_result.match_to_comparison(
       f.as_emma_model(),title='COMPARE',
       tolerance= search_result.search_params.comparison_emma_model_tolerance,
       display=False,out=out)[0]
    if matching is None or n_match >= matching \
        or n_match == f.scatterers().size():
       new_fragments.append(f)
  print >>out,"Selected %d of %d fragments matching comparison model at %d sites" %(len(new_fragments),len(fragments),matching)
  return new_fragments
  

def print_results(search_result=None,fragments=[],structures=[],
    final_scores=[],out=sys.stdout):
  i=0
  if search_result and search_result.search_params.comparison_emma_model:
    print >>out,"Fragment   Sites   Score  Matches"
  else:
    print >>out,"Fragment   Sites   Score"
  if structures and final_scores:
    zip_list=[]
    for f,score in zip(structures,final_scores):
      zip_list.append([f,score.score()])
  elif fragments:
    zip_list=[]
    for f in fragments:
      zip_list.append([f,getattr(f,'score',0.)])
  else:
    zip_list=[]
   
  for structure,score in zip_list:
    if search_result and search_result.search_params.comparison_emma_model:
      n_match=search_result.match_to_comparison(
          structure.as_emma_model(),title='COMPARE',
          tolerance=
            search_result.search_params.comparison_emma_model_tolerance,
             display=False,out=out)
      print >>out,"   %d       %d    %7.3f    %d" %(
        i,structure.scatterers().size(),score,n_match[0])
      if n_match[0] == structure.scatterers().size(): 
        print >>out,structure.as_pdb_file() 
    else:
      print >>out,"   %d       %d    %7.3f " %(
        i,structure.scatterers().size(),score)
    i+=1
  print >>out

def sift_fragments(substructure_params=None,
   tolerance=None,
   fragments=[],
   structures=[],final_scores=[],
   n_keep=None,
   out=sys.stdout):
  list_of_models=[]
  list_of_unique_fragments=[]
  list_of_unique_scores=[]
  # work down n_keep of the models and keep only unique ones
  zip_list=[]
  if fragments:
    for f in fragments:
      zip_list.append([f,0.])
  else:
    for structure,score in zip(structures,final_scores):
      zip_list.append([structure,score])
  for fragment,score in zip_list:
    n_sites=fragment.sites_cart().size()
    emma_model=fragment.as_emma_model()
    n_match=match_to_list_of_models(substructure_params=substructure_params,
     tolerance=tolerance,list_of_models=list_of_models,
     emma_model=emma_model,out=out)
    if n_match < n_sites:  # unique
      list_of_models.append(emma_model)
      list_of_unique_fragments.append(fragment)
      list_of_unique_scores.append(score)
    if n_keep is not None and len(list_of_unique_fragments)>=n_keep: break
  return list_of_unique_fragments,list_of_unique_scores

def match_to_list_of_models(substructure_params=None,
   tolerance=None,list_of_models=[],emma_model=None,
   out=sys.stdout):
  from cctbx import euclidean_model_matching as emma
  if tolerance is None:
     tolerance=0.1* substructure_params.min_distance
  best_n=0
  for x in list_of_models:
      assert x.unit_cell().is_similar_to(emma_model.unit_cell())
      comparison_top_matches = emma.model_matches(
        model1 = emma_model,
        model2 = x,
        tolerance = tolerance,
        models_are_diffraction_index_equivalent = True,
        break_if_match_with_no_singles=True
        )
      n=comparison_top_matches.n_pairs_best_match()
      if n>best_n: 
        best_n=n
  return best_n

def run_brute_force(
   n_iter=1,
   seed_text="PATTERSON",
   cb_op_niggli=None,
   best_search_result=None,
   fragment_list=None,
   fragment_list_is_scored=False,
   high_resolution_cutoff=None,
   previous_search_result=None,
   orig_search_params=None,orig_params=None,
   orig_substructure_params=None,scatterer=None,
       original=None,anom_diffs=None,out=sys.stdout):

  print >>out,\
    "\n ---  RUNNING BRUTE FORCE WITH %s SEEDING, ITERATION %d --- \n" %(
     seed_text,n_iter)
  # start out with original parameters
  from copy import deepcopy
  search_params=deepcopy(orig_search_params) 
  params=deepcopy(orig_params)
  substructure_params=deepcopy(orig_substructure_params)

  # set parameters for this run

  search_result=None

  if high_resolution_cutoff:
    params.resolution=high_resolution_cutoff

  if not fragment_list:
    return best_search_result,False,previous_search_result # nothing to do

  n_iter_max=params.brute_force_completion.n_iter_max
  n_keep=params.brute_force_completion.n_top_iter
  if n_iter==1:
    print >>out,"Keeping %d solutions in each of %d iterations." %(
      n_keep,n_iter_max)
 
  search_params.tries_to_skip=0
  #search_params.max_tries should be max_tries_per_processor

  if not fragment_list_is_scored:
    # Sort our Patterson solutions (score them all).
    # Normally this is phaser-complete scoring but we are not going to complete anything
    search_params.score_only=True
    print >>out,"Scoring %d fragments now and taking top %d" %(
      len(fragment_list),
      min(params.brute_force_completion.n_top_patt,len(fragment_list)))
    from copy import deepcopy
    search_result=run_group(params=params,
       search_params=deepcopy(search_params),
       substructure_params=deepcopy(substructure_params),
       scatterer=scatterer,
       fragment_list=fragment_list,
       llgc_sigma=1000.,
       max_structures=len(fragment_list), # take all structures here
       original=original,anom_diffs=anom_diffs,out=out)
    working_fragment_list=get_fragment_list(
        search_result=search_result,cb_op_niggli=cb_op_niggli)
    print >>out,"SORTED %s LIST with %d entries:" %(
       seed_text,len(working_fragment_list))
  else:
    working_fragment_list=fragment_list
    print >>out,"WORKING FRAGMENT LIST:"

  if params.brute_force_completion.select_matching: # select those matching
    working_fragment_list=select_good(working_fragment_list,search_result, 
     matching=2,out=out) 

  working_fragment_list,dummy_scores=sift_fragments(
   substructure_params=substructure_params,
   fragments=working_fragment_list,
   n_keep=params.brute_force_completion.n_top_patt,
   out=out)

  print_results(search_result=search_result,fragments=working_fragment_list,
    out=out)

  structures=[]
  final_scores=[]
  # for each working solution try to complete it
  all_fragments=[]
  dummy_search_result=search_result
  search_result=None
  print >>out, "Creating LLG maps based on each of the top %d fragments." %(
     len(working_fragment_list))
  print >>out, "For each LLG map, taking all combinations of %d of top %d " %(
   params.brute_force_completion.n_llg_add_at_once,
   params.brute_force_completion.n_top_llg) + " peaks "
  print >>out,"and adding them to the starting fragments."
  for f in working_fragment_list:
    new_fragment_list=run_llg_brute_force(
      params=params,search_params=search_params,
      cb_op_niggli=cb_op_niggli,
      substructure_params=substructure_params,
      scatterer=scatterer,
      fragment=f,
      dummy_search_result=dummy_search_result,
      original=original,anom_diffs=anom_diffs,out=out)

    # Can select only the good ones here. Nothing done if select_matching=None
    if params.brute_force_completion.select_matching:
      new_fragment_list=select_good(new_fragment_list,dummy_search_result, 
       matching=params.brute_force_completion.select_matching,out=out)

    print >>out,"NEW SEARCH LIST"
    print_results(search_result=dummy_search_result,fragments=new_fragment_list,
      out=out)

    # and now score this new fragment list

    print >>out,"Total of %d fragments to score" %(len(new_fragment_list))
    out.flush()
    if len(new_fragment_list) < 1:  # skip it
      continue

    search_params.score_only=True
    if params.brute_force_completion.score_additions_with_completion:
      llgc_sigma=None # use original value
    else:
      llgc_sigma=1000.
    search_result=run_group(params=params,
      search_params=deepcopy(search_params),
      substructure_params=deepcopy(substructure_params),
      scatterer=scatterer,
      fragment_list=new_fragment_list,
      llgc_sigma=llgc_sigma,
      max_structures=None, # take all structures here
      original=original,anom_diffs=anom_diffs,out=out)
    structures+=search_result.structures
    final_scores+=search_result.final_scores
    print >>out,"Added %d results..." %(len(search_result.structures))
    out.flush()

  if not structures:
    print >>out,"\nNO SCORING RESULTS"
    return previous_search_result,False,previous_search_result

  structures,final_scores=sort_structures(structures,final_scores) #, n_keep=n_keep)

  print >>out,"\nSORTED SCORING RESULTS"
  out.flush()
  print_results(search_result=search_result,structures=structures,
    final_scores=final_scores,out=out)

  structures,final_scores=sift_fragments(
   substructure_params=substructure_params,
   structures=structures,final_scores=final_scores,
   n_keep=n_keep,
   out=out)
  print >>out,"\nSORTED SIFTED SCORING RESULTS"
  print_results(search_result=search_result,structures=structures,
    final_scores=final_scores,out=out)

  # and tack these structures on to a search object to return
  search_result.structures=structures
  search_result.final_scores=final_scores
  search_result.fully_recycled_flags=len(structures)*[True]
  search_result.extrapolation_scan_scores=len(structures)*[True]

  quit=False
  if n_iter >= n_iter_max:
    print >>out,"Stopping iterations as maximum of %d reached" %(n_iter_max)
    quit=True
  elif  (structures[0].sites_cart() and 
     len(structures[0].sites_cart()) > substructure_params.n_sites):
    print >>out,"Stopping iterations as sufficient sites found (%d)" %(
      len(structures[0].sites_cart()))
    quit=True
  if quit:  # get consensus model and return
    #search_result.search_params.min_top_n_matches=1
    search_result.update_top_group()
    search_result.euclidean_model_matching(out=out)

    best_consensus_only = (search_result.top_matches is not None)

    if (search_result.top_matches is not None):
      search_result.show_top_matches(True,out=out)
      search_result.minimize_consensus_model(
        n_sites=search_result.substructure_params.n_sites,out=out)

    return search_result,False,previous_search_result # done


  else:  # run again
    fragment_list=get_fragment_list(
      search_result=search_result,cb_op_niggli=cb_op_niggli)

    search_result,done,previous_search_result=\
    run_brute_force(
          fragment_list_is_scored=True,
          cb_op_niggli=cb_op_niggli,
          seed_text="BRUTE FORCE",
          high_resolution_cutoff=high_resolution_cutoff,
          fragment_list=fragment_list,
          best_search_result=best_search_result,
          orig_search_params=orig_search_params,
          orig_params=orig_params,
          orig_substructure_params=orig_substructure_params,
          scatterer=scatterer,original=original,anom_diffs=anom_diffs,
          n_iter=n_iter+1,
          out=out)
    return search_result,False,previous_search_result # done

def sort_structures(structures,final_scores,n_keep=None):
  sort_list=[]
  for structure,score in zip(structures,final_scores):
    sort_list.append([score.score(),score,structure])
  sort_list.sort()
  sort_list.reverse()
  new_structures=[]
  new_final_scores=[]
  i=0
  for dummy_score,score,structure in sort_list:
    i+=1
    new_structures.append(structure)
    new_final_scores.append(score)
    if n_keep is not None and i >= n_keep: break

  return new_structures,new_final_scores

def sort_fragments(fragments):
  new_list=[]
  for f in fragments:
    new_list.append([f.score,f])
  new_list.sort()
  new_list.reverse()
  sorted_fragments=[]
  sort_id=0
  for score,f in new_list:
    f.sort_id=sort_id
    sorted_fragments.append(f)
    sort_id+=1
  return sorted_fragments

def get_mean_b_iso_and_occ(structure):
  # print structure.as_pdb_file()
  # just return mean b_iso and occ
  bmin,bmax,b_iso=structure.b_iso_min_max_mean()
  occ=structure.scatterers().extract_occupancies().min_max_mean().mean
  return b_iso,occ

def remove_close(llg_peak_list,llg_height_list,min_cross_distance=1.2,
    unit_cell=None,n_keep=None):
  from scitbx.matrix import col
  new_peak_list=[]
  new_height_list=[]
  maxdd=min_cross_distance*min_cross_distance
  for xyz,h in zip(llg_peak_list,llg_height_list):
    same=False
    xyz_orth=unit_cell.orthogonalize(xyz)
    for xx in new_peak_list:
      xx_orth=unit_cell.orthogonalize(xx)
      dd=col(xyz_orth)-col(xx_orth)
      if dd.norm_sq() < maxdd:
        same=True
        break
    if not same:
      new_peak_list.append(xyz)
      new_height_list.append(h)
      if n_keep is not None and len(new_peak_list)>=n_keep: break
 
  return new_peak_list,new_height_list

def run_llg_brute_force(
      params=None,search_params=None,
      substructure_params=None,
      scatterer=None,
      cb_op_niggli=None,
      fragment=None,
      dummy_search_result=None,
      original=None,anom_diffs=None,out=sys.stdout):
    
    # get LLG map based on fragment; find peaks, make new possible solutions
    # with brute force addition and score them all

    # get the LLG map and find all the peaks
    # max_peaks=params.brute_force_completion.n_top_llg
    # peak_cutoff=params.brute_force_completion.min_sigma_in_brute_force


    llg_peak_list,llg_height_list=run_llg_map(
        params=params,search_params=search_params,
      substructure_params=substructure_params,
      scatterer=scatterer,
      cb_op_niggli=cb_op_niggli,
      fragment=fragment,
      min_cross_distance=0.5* substructure_params.min_distance, 
      original=original,anom_diffs=anom_diffs,out=out)
  
    # Remove close ones (should have happened in peak search...) and cut off
    print >>out,"Done getting LLG map with %d peaks " %(len(llg_peak_list))
    out.flush()

    llg_peak_list,llg_height_list=remove_close(llg_peak_list,llg_height_list,
       min_cross_distance=0.5* substructure_params.min_distance,
       unit_cell=original.unit_cell(),
       n_keep=params.brute_force_completion.n_top_llg)
 
    print >>out,"Done sifting peaks, %d remaining" %(len(llg_peak_list))
    out.flush()
    # add groups of sites
    new_fragments=[]

    b_iso,occupancy=get_mean_b_iso_and_occ(fragment)
    from cctbx.array_family import flex
    from cctbx import xray
    base_scatterers = flex.xray_scatterer()
    if params.brute_force_completion.include_base_scatterers:
      for site in fragment.sites_frac():
        base_scatterers.append(
          xray.scatterer(scattering_type=scatterer.scattering_type,
          label=scatterer.element_symbol(), site=site, 
        ))
    from copy import deepcopy
    n_peaks=len(llg_peak_list)
    addition_list=get_addition_list(
       params.brute_force_completion.n_llg_add_at_once,n_peaks,out=out)

    model_number=0
    xray_scatterer=xray.scatterer( scattering_type =scatterer.scattering_type)
    for add in addition_list:
      model_number+=1
      scatterers=deepcopy(base_scatterers)
   
      sum_heights=0. 
      for i in add:
        site=llg_peak_list[i]
        sum_heights+=(llg_height_list[i]-llg_height_list[-1])
        scatterers.append(
           xray.scatterer(scattering_type=scatterer.scattering_type,
           label=scatterer.element_symbol(), site=site, 
        ))
      f=xray.structure(fragment.xray_structure(), scatterers=scatterers)
      f.set_b_iso(b_iso)
      f.set_occupancies(occupancy)
      f.model_number=model_number
      f.i_patterson_vector=model_number
      f.i_fragment=0
      f.score_value=sum_heights
      f.score=sum_heights
      new_fragments.append(f)
    print >>out,"Ready with %d trial fragments" %(len(new_fragments))
    out.flush()
    return new_fragments

def get_addition_list(n_llg_add_at_once,n_peaks,out=sys.stdout):
  # figure out what peaks to add based on n_llg_add_at_once
  addition_list=[]
  if n_llg_add_at_once==1:
    for i in xrange(n_peaks):
      addition_list.append([i])
  elif n_llg_add_at_once==2:
    for i in xrange(n_peaks-1):
      for j in xrange(i+1,n_peaks):
        addition_list.append([i,j])
  elif n_llg_add_at_once==3:
    for i in xrange(n_peaks-2):
      for j in xrange(i+1,n_peaks-1):
        for k in xrange(j+1,n_peaks):
          addition_list.append([i,j,k])
  elif n_llg_add_at_once>=4:  # no way anyone will want more
    for i in xrange(n_peaks-3):
      for j in xrange(i+1,n_peaks-2):
        for k in xrange(j+1,n_peaks-1):
          for l in xrange(k+1,n_peaks):
            addition_list.append([i,j,k,l])
  return addition_list 

def run_llg_map(params=None,search_params=None,
      substructure_params=None,
      scatterer=None,
      fragment=None,
      cb_op_niggli=None,
      min_cross_distance=1.2,
      original=None,anom_diffs=None,out=sys.stdout):
   peak_list=[]

   from phaser.phenix_adaptors import hyss_scoring
   from libtbx.utils import null_out
   from copy import deepcopy
   scanner=hyss_scoring.phaser_map_based_extrapolation(
     hyss_search=run_group(dummy_search=True,
       params=params,
       search_params=deepcopy(search_params),
       substructure_params=deepcopy(substructure_params),
       scatterer=scatterer,
       original=original,anom_diffs=anom_diffs,out=null_out()),
     out=null_out()) 
   n_sites=None
   (llg,llg_map)=scanner.get_fft_map(fragment=fragment)
   llg_map.apply_sigma_scaling() # you can sigma-scale map too
   map_data=llg_map.real_map_unpadded() 
   from cctbx import maptbx
   peak_search_parameters = maptbx.peak_search_parameters(
       peak_search_level      = 1,
       max_peaks              = int(
           1.5*float(params.brute_force_completion.n_top_llg)),
       peak_cutoff            = params.brute_force_completion.min_sigma_in_brute_force,
       interpolate            = True,
       min_distance_sym_equiv = None,
       general_positions_only = False,
       min_cross_distance     = min_cross_distance, # distance between peaks to consider tham separate
       min_cubicle_edge       = 5)
   crystal_gridding_tags = llg_map.tags()
   cluster_analysis = crystal_gridding_tags.peak_search(
       parameters = peak_search_parameters,
       map        = map_data)

   return cluster_analysis.sites(),cluster_analysis.heights()

def run_phaser_completion_on_existing_solutions(
   seed_text="PATTERSON",
   cb_op_niggli=None,
   best_search_result=None,
   fragment_list=None,
   high_resolution_cutoff=None,
   starting_seed=None,
   seeds_to_use=None,
   previous_search_result=None,
   orig_search_params=None,orig_params=None,
   orig_substructure_params=None,scatterer=None,
       original=None,anom_diffs=None,out=sys.stdout):

  print >>out,"\n ---  RUNNING PHASER COMPLETION WITH %s SEEDING --- \n" %(
     seed_text)
  # start out with original parameters
  from copy import deepcopy
  search_params=deepcopy(orig_search_params) 
  params=deepcopy(orig_params)
  substructure_params=deepcopy(orig_substructure_params)

  # set parameters for this run

  params.rescore='phaser-complete'
  search_params.score_only=True
  if high_resolution_cutoff:
    params.resolution=high_resolution_cutoff

  if fragment_list and starting_seed:  # starts with 0
    fragment_list=fragment_list[starting_seed:]
  if fragment_list and seeds_to_use:
    fragment_list=fragment_list[:seeds_to_use]

  if not fragment_list:
    return best_search_result,False,previous_search_result # nothing to do

  groups,max_tries_per_processor=get_groups(
    params=params,search_params=search_params,
    substructure_params=substructure_params,
    fragment_list=fragment_list,
    out=out)

  search_params.tries_to_skip=0
  for i in xrange(groups):
    if groups > 1:
      print >>out,"\n--- GROUP %d OF %d OF PHASER COMPLETION WITH %s SEEDING ---\n" %(i+1,groups,seed_text)
    if previous_search_result and previous_search_result.structures:
      print "Including %d previous structures from Phaser " %(
         len(previous_search_result.structures))+\
       "completion with \n%s seeding at this resolution" %(seed_text.lower())
    search_result=run_group(params=params,search_params=search_params,
     substructure_params=substructure_params,
     scatterer=scatterer,
     previous_search_result=previous_search_result,
     fragment_list=fragment_list,
     original=original,anom_diffs=anom_diffs,out=out)

    if search_result.failed: continue

    if search_result:
      previous_search_result=hyss.subprocess_search_result(
       search_result,
          max_structures=params.auto_control.solutions_to_save)
      best_search_result=get_best_search_result(
        best_search_result,search_result=search_result,
        scatterer=scatterer,params=params,out=out)

    if search_result.terminated_early:
      print >>out,"\nFinished with search...\n"
      print >>out,"\n--- END OF GROUP %d OF %d OF PHASER COMPLETION WITH %s SEEDING ---\n" %(i+1,groups,seed_text)
      return best_search_result,True,previous_search_result  # done
 
    # Otherwise...try more patterson vectors... 

    search_params.tries_to_skip=search_params.max_tries # skip what we did
    search_params.max_tries+=max_tries_per_processor          # and do next set 
    if groups > 1:
      print >>out,"\n--- END OF GROUP %d OF %d OF PHASER COMPLETION WITH %s SEEDING ---\n" %(i+1,groups,seed_text)


  if search_result.terminated_early:
    print >>out,"\nFinished with search...\n"
    return best_search_result,True,previous_search_result  # done
  else:
    return best_search_result,False,previous_search_result# done

def run_standard(best_search_result=None,
   patterson_seeds_only=False,
   high_resolution_cutoff=None,
   fragment_list=None,
   seeds_to_use=None,
   starting_seed=None,
   allow_shake=False,
   previous_search_result=None,
   orig_search_params=None,orig_params=None,
   orig_substructure_params=None,scatterer=None,
       original=None,anom_diffs=None,out=sys.stdout):

  # start out with original parameters
  from copy import deepcopy
  search_params=deepcopy(orig_search_params) 
  params=deepcopy(orig_params)
  substructure_params=deepcopy(orig_substructure_params)
 
  # set parameters for this run
  if not allow_shake:
    search_params.n_shake=0
  if high_resolution_cutoff:
    params.resolution=high_resolution_cutoff

  if patterson_seeds_only:
    print >>out,"\n ---  GETTING PATTERSON SEEDS --- \n" 
    print >>out,"High-resolution limit: %7.2f A " %(
       search_params.high_resolution_cutoff)
    search_params.max_tries=-1 # take all patterson peaks
    search_params.unit_occupancy=True # return all occupancies as 1.0
    params.rescore='correlation' # always for patterson seeds
    params.nproc=1  # don't need multiprocessing
    max_tries_per_processor=None
    if params.verbose:
      local_out=out
    else:
      from libtbx.utils import null_out
      local_out=null_out()

    if search_params.input_emma_model_list:
      print >>out,"\nUSING INPUT EMMA MODELS INSTEAD OF PATTERSON SEEDS\n"
      replace_patterson_with_input_emma_models=True
      print >>out, "\nInput emma models:"
      xray_scatterer=xray.scatterer(scattering_type=scatterer.scattering_type)
      iii=0
      for m in search_params.input_emma_model_list:
        iii+=1
        print >>out,"Model %d" %(iii)
        print >>out,m.as_xray_structure(xray_scatterer).as_pdb_file()
      print >>out
    
    else:
      replace_patterson_with_input_emma_models=False
  
    search_result=run_group(params=params,search_params=search_params,
       substructure_params=substructure_params,
       return_unsorted_patterson_solutions=True,
       replace_patterson_with_input_emma_models=\
           replace_patterson_with_input_emma_models,
       scatterer=scatterer,
       fragment_list=fragment_list,
       previous_search_result=previous_search_result,
       original=original,anom_diffs=anom_diffs,out=local_out)
    print >>out,"Total of %d Patterson seeds identified" %(
        len(search_result.unsorted_patterson_solutions))
    return search_result.unsorted_patterson_solutions,search_result.cb_op_niggli

  else:
    if fragment_list and starting_seed:  # starts with 0
      fragment_list=fragment_list[starting_seed:]
    if fragment_list and seeds_to_use:
      fragment_list=fragment_list[:seeds_to_use]
    if len(fragment_list)==0:
      print >>out,"No seeds to test"
      return None,False,previous_search_result  #  just quit
     

    print >>out,"\n ---  RUNNING DIRECT METHODS WITH PATTERSON SEEDS --- \n" 

    groups,max_tries_per_processor=get_groups(
      fragment_list=fragment_list,
      params=params,search_params=search_params,
      substructure_params=substructure_params,
      out=out)
    local_out=out

  for i in xrange(groups):
    print >>out,\
       "\n--- GROUP %d OF DIRECT METHODS WITH PATTERSON SEEDING ---\n" %(i+1)

    if previous_search_result and previous_search_result.structures:
      print "Including %d previous structures from direct methods at this resolution" %(
         len(previous_search_result.structures))
    params.rescore='correlation'
    search_result=run_group(params=params,search_params=search_params,
       substructure_params=substructure_params,
       scatterer=scatterer,
       fragment_list=fragment_list,
       previous_search_result=previous_search_result,
       original=original,anom_diffs=anom_diffs,out=local_out)

    if search_result:
      previous_search_result=hyss.subprocess_search_result(
       search_result,max_structures=params.auto_control.solutions_to_save)

      best_search_result=get_best_search_result(
        best_search_result=best_search_result,search_result=search_result,
        scatterer=scatterer,params=params,out=local_out)

    if search_result.terminated_early:
      print >>out,"\nFinished with search...\n"
      print >>out,\
       "\n--- END OF GROUP %d OF DIRECT METHODS WITH PATTERSON SEEDING ---\n" %(i+1)
      return best_search_result,True,previous_search_result  # done
 
    # Otherwise...try more patterson vectors... 

    search_params.tries_to_skip=search_params.max_tries # skip what we did
    search_params.max_tries+=max_tries_per_processor          # and do next set 
    if groups > 1:
      print >>out,\
       "\n--- END OF GROUP %d OF DIRECT METHODS WITH PATTERSON SEEDING ---\n" %(i+1)


  return best_search_result,False,previous_search_result  # not done

def get_best_search_result(best_search_result=None,search_result=None,
    scoring_search_result=None,
    params=None,
    scatterer=None,out=sys.stdout):
  if not best_search_result or not best_search_result.consensus_model:
    best_search_result=search_result
    print >>out,"Taking current result as new best (no existing best)"
  if not best_search_result or not best_search_result.consensus_model:
    return None

  if not search_result or search_result.top_matches is None or \
      search_result.top_matches.n_matches()== 0: 
    return best_search_result

  print >>out,"\n Identifying current best solution...\n"

  assert best_search_result is not None

  # decide which one to score with:
  if scoring_search_result is None: 
    best_is_phaser=best_search_result.scoring.scoring_type in [
      'phaser-complete','phaser-refine']
    search_result_is_phaser=search_result.scoring.scoring_type in [
      'phaser-complete','phaser-refine']

    if best_is_phaser and not search_result_is_phaser:
      scoring_search_result=best_search_result
    elif search_result_is_phaser and not best_is_phaser:
      scoring_search_result=search_result
    else:  # take the one with the higher resolution
      if best_search_result.resolution_used <= \
           search_result.resolution_used:
        scoring_search_result=best_search_result
      else:
        scoring_search_result=search_result

  print >>out,"\nScoring basis: %s at %7.1f A\n" %(
   scoring_search_result.scoring.scoring_type,
     scoring_search_result.resolution_used)

  # if scoring_search_result is different than search_result, rescore..
  # use best search result to score current search result
  # first convert consensus model to niggli cell

  try:
    search_result.consensus_score=\
      scoring_search_result.refinement.minimize(structure=
      search_result.consensus_model.as_emma_model()
       .as_xray_structure(search_result.xray_scatterer )
       .change_basis( search_result.cb_op_niggli ) )
  except Exception,e:
    print >>out,"Warning consensus model scoring failed ...carrying on. \n"
    from phenix.substructure.hyss import correlation_coefficient_score
    search_result.consensus_score=correlation_coefficient_score(-99999.9)

  print >>out, "\nResults from search: Score: %8.4f" %(
       search_result.consensus_score.score())

  s=consensus_model_as_pdb_string(consensus_model=search_result.consensus_model,
    scatterer=scatterer)
  print >>out, "\nConsensus model: \n\n",s

  if scoring_search_result.scoring.scoring_type == 'correlation':
    similar_score=0.02
  else:
    similar_score=1.0

  if search_result.terminated_early and \
      not best_search_result.terminated_early and \
      search_result.consensus_score.score() > \
        best_search_result.consensus_score.score()-similar_score:
    # use search_result instead of best_search_result
    print >>out, \
       "Keeping search result as it terminated early and is similar"+\
       "\nin score to best result\n"
    take_search_result=True
  else:
    take_search_result=None
    
  if take_search_result or \
     not best_search_result or \
     search_result.consensus_score > best_search_result.consensus_score:
    if best_search_result:
      print >>out,"Previous best score: %8.4f " %(
        best_search_result.consensus_score.score())
    best_search_result=search_result

    print >>out,"New best search result at resolution of %7.2f A with score of %8.4f" %(best_search_result.resolution_used,
      best_search_result.consensus_score.score())
    print >>out,"Scoring basis: %s\n" %(
      scoring_search_result.scoring.scoring_type)
  elif search_result is best_search_result: # first time
    print >>out,"Current search result at resolution of %7.2f A kept with score of %8.4f" %(best_search_result.resolution_used,
      best_search_result.consensus_score.score())
  else: # previous was better
    print >>out,"Previous search result at resolution of %7.2f A kept with score of %8.4f" %(best_search_result.resolution_used,
      best_search_result.consensus_score.score())

  if params.auto_control.solutions_to_save and \
    len(best_search_result.structures)> params.auto_control.solutions_to_save:
    print >>out,"Taking top %d structures (of %d)" %(
     params.auto_control.solutions_to_save,len(best_search_result.structures))
    best_search_result.get_top_solutions(
         max_structures=params.auto_control.solutions_to_save)
  return best_search_result

  
def run_group(params=None,search_params=None,
  substructure_params=None,
  previous_search_result=None,
  return_unsorted_patterson_solutions=False,
  replace_patterson_with_input_emma_models=False,
  fragment_list=None,
  scatterer=None,
  standard_run=False,
  original=None,anom_diffs=None,
  max_structures=None,
  rescore=None,
  dummy_search=False,
  llgc_sigma=None,
  out=sys.stdout):

  if params.verbose or standard_run:
    local_out=out
  else:
    from libtbx.utils import null_out
    local_out=null_out()

  # Apply resolution cutoffs if necessary
  original_with_phaser_resol_cutoff,anom_diffs_with_cutoff=\
       apply_resolution_cutoffs(
    resolution=params.resolution,
    low_resolution=params.low_resolution,
    phaser_resolution=params.phaser_resolution,
    phaser_low_resolution=params.phaser_low_resolution,
    original=original,anom_diffs=anom_diffs,out=local_out)
  # ready with data

  # say what we are going to do:
  if fragment_list:
    print >>out,"Starting with working fragment list"
  elif search_params.input_emma_model_list:
    print>>out, "Using input models from input_emma_model_list"
  else:  
    print>>out, "Getting two-site solutions from Patterson function"
  if search_params.score_only:
    print>>out, "Scoring solutions (%s scoring)"  %(params.rescore)
  print >>out
 
  if llgc_sigma is None:
    llgc_sigma=params.llgc_sigma
  if max_structures is None:
    max_structures=params.auto_control.solutions_to_save
  if rescore is None:
    rescore=params.rescore

  search_result = hyss.search(
    f_obs=anom_diffs_with_cutoff,
    substructure_params=substructure_params,
    search_params=search_params,
    f_original = original_with_phaser_resol_cutoff,
    scattering_type = scatterer.scattering_type,
    wavelength = params.wavelength,
    llgc_sigma = llgc_sigma,
    rescore = rescore,
    extrapolation = params.direct_methods.extrapolation,
    nproc=params.nproc,
    previous_search_result=previous_search_result,
    fragment_list=fragment_list,
    max_structures=max_structures,
    return_unsorted_patterson_solutions=return_unsorted_patterson_solutions,
    replace_patterson_with_input_emma_models=replace_patterson_with_input_emma_models,
    dummy_search=dummy_search,
    out=out,
    )
  search_result.resolution_used=params.resolution # tack it on so we have it

  return search_result

def get_fragment_list(search_result=None,cb_op_niggli=None):
  structure_list=[]
  from copy import deepcopy
  for structure,score in zip(
       search_result.structures,search_result.final_scores):
    structure_list.append((score.score(),score.correlation(),
        structure))
  structure_list.sort()
  structure_list.reverse()
  fragment_list=[]
  sort_id=-1
  for score,correlation_score,structure in structure_list:
    sort_id+=1
    fragment=structure.change_basis( cb_op_niggli) 
    fragment.i_patterson_vector=structure.info.i_patterson_vector
    fragment.i_fragment=structure.info.i_fragment
    fragment.model_number=structure.model_number
    fragment.sort_id=sort_id  # this should be a continuous series of integers
                              # if no sort_id, then model_number is used.
    fragment.correlation_score=correlation_score
    fragment.score=score
    fragment_list.append(fragment)
  return fragment_list


def get_groups(params=None,
   search_params=None,
   substructure_params=None,
   fragment_list=None,
   out=sys.stdout):


  from copy import deepcopy
  search_params_default=deepcopy(search_params) 
  search_params_default.compute_derived(n_sites=substructure_params.n_sites)

  # split fragment_list into groups of 2*max(nproc,n_top_models_to_compare) 
  #  for evaluation

  total_fragments=len(fragment_list)
  group_size=2*max(params.nproc,search_params_default.n_top_models_to_compare)
  group_size=max(1,min(group_size,total_fragments))
  groups=(total_fragments+group_size-1)//group_size
  groups=max(1,min(params.auto_control.max_groups,groups))
  group_size=(total_fragments+groups-1)//groups
  tries_per_processor=(group_size+params.nproc-1)//params.nproc
 
  search_params.max_tries=tries_per_processor  # tries per processor per gruop

  print >>out, " Number of seeds: %d "%(total_fragments)
  print >>out," Evaluations per processor per group: %d   nproc: %d"%(
       tries_per_processor,params.nproc)
  print >>out," Total tries in each group: %d  Total groups %d  " %(
      group_size,groups,)

  print >>out," High-resolution limit: %7.2f "%(params.resolution)

  if fragment_list and hasattr(fragment_list[0],'correlation_score'):
    print >>out,\
      "\n Starting with %d seeds with scores from %7.2f to %7.2f" %(
      total_fragments,
      fragment_list[-1].correlation_score,
      fragment_list[0].correlation_score)
  if fragment_list and params.verbose:
      xray_scatterer=xray.scatterer(
         scattering_type =search_params.scattering_type)
      iii=-1
      for fragment in fragment_list:
        iii+=1
        print "MODEL %d i_patt %d" %(iii,fragment.i_patterson_vector),
        fragment.as_emma_model().as_xray_structure(xray_scatterer).as_pdb_file()

  return groups,tries_per_processor

def apply_resolution_cutoffs(out=sys.stdout,
   resolution=None,
   low_resolution=None,
   phaser_resolution=None,
   phaser_low_resolution=None,
   original=None,
   anom_diffs=None):
  print >>out,"\nAPPLYING RESOLUTION CUTOFFS\n"
  if phaser_resolution is None: phaser_resolution=resolution

  # set resolution to d_min+0.5 if never set

  if resolution is None and phaser_resolution is None:
    if original is not None:  # 2014-05-05
      orig_resolution=original.d_min()
    else:
      orig_resolution=anom_diffs.d_min()
    resolution=orig_resolution+0.5
 
    
    phaser_resolution=resolution
    print >>out,\
     "Set resolution to %8.2f A as it is not set and d_min is %8.2f A" %(
        resolution,orig_resolution)

  if phaser_resolution is not None and original is not None:
    if phaser_low_resolution is None: phaser_low_resolution=10000.
    print >>out,"Setting resolution for phaser to %8.2f to %8.2f A\n" %(
      phaser_low_resolution,phaser_resolution)
    original_with_phaser_resol_cutoff  = resolution_cutoff.process(
        observations=original,
        d_max=phaser_low_resolution,
        d_min=phaser_resolution,
        sigma_cutoff=None, # 2014-05-20 put this in to keep everything
        out=out)
  else:
    original_with_phaser_resol_cutoff  = original

  if resolution or low_resolution:
    if low_resolution is None:
      low_resolution=15.0
      print >>out,"(Set low-resolution limit by default to 15 A)"
    print >>out,"\nSetting resolution for Patterson and direct methods to: ",
    print >>out,"%7.2f A - " %(low_resolution),
    if resolution: 
      print >>out,"%7.2f A " %(resolution)
    else:
      print >>out,"None "
    anom_diffs_with_cutoff= resolution_cutoff.process(
        observations=anom_diffs,
        d_max=low_resolution,
        d_min=resolution,
        sigma_cutoff=None,
        out=out)
    print >>out
  else:
    anom_diffs_with_cutoff=anom_diffs
  return original_with_phaser_resol_cutoff,anom_diffs_with_cutoff

def process_scattering_type(scattering_type):
  for entry in eltbx.tiny_pse.table_iterator():
    if scattering_type.strip().lower() == entry.name():
      scattering_type = entry.symbol()
      break

  scattering_type = eltbx.xray_scattering.get_standard_label(
    label=scattering_type)
  scatterer = xray.scatterer(scattering_type=scattering_type, b=20)
  if (scatterer.scattering_type in ("S", "Se")):
    general_positions_only = True
  else:
    general_positions_only = False
  return scatterer, general_positions_only

def match_data_label(input_arrays, data_label, d_max=None, d_min=None,
    rms_cutoff=None,
    sigma_cutoff=None,
    out=sys.stdout):
  label_table = reflection_file_utils.label_table(
    miller_arrays=input_arrays)
  selected_array = label_table.match_data_label(
    label=data_label,
    command_line_switch="--data_label")
  if (selected_array is None):
    raise Sorry("Please try again.")
  if not d_min: d_min=selected_array.d_min() #2014-01-02
  if (is_experimental_anomalous_array(selected_array)):
    anom_diffs = anomalous_diffs.process(
      observations=selected_array,
      d_max=d_max,
      d_min=d_min,
      rms_cutoff=rms_cutoff,
      sigma_cutoff=sigma_cutoff,
      out=out)
    anom_diffs.set_info(
      "Externally selected. anomalous signal: %.3f" % anom_diffs.info())
    data = ( selected_array, anom_diffs )
  elif (is_anomalous_difference_array(selected_array)):
    anom_diffs = resolution_cutoff.process(
      observations=selected_array,
      d_max=d_max,
      d_min=d_min,
      out=out)
    anom_diffs.set_info(
      "Externally selected, externally preprocessed." % anom_diffs.info())
    data = ( None, anom_diffs )
  else:
    print >>out
    print >>out,"The input array with the label %s" % selected_array.info()
    print >>out,"does not contain suitable data."
    print >>out
    raise Sorry("Unsuitable data array selected.")
  return data

def compare_anom_diff_arrays(a, b):
  ar = isinstance(a[0].observation_type(),
    cctbx.xray.observation_types.reconstructed_amplitude)
  br = isinstance(b[0].observation_type(),
    cctbx.xray.observation_types.reconstructed_amplitude)
  if (not ar and br): return -1
  if (ar and not br): return  1
  ai = a[0].is_xray_intensity_array()
  bi = b[0].is_xray_intensity_array()
  if (not ai and bi): return -1
  if (ai and not bi): return  1
  if (a[1].info() > b[1].info()): return -1
  if (a[1].info() < b[1].info()): return  1
  return 0

def select_array_to_use(input_arrays, d_max=None, d_min=None,
    rms_cutoff=None,
    sigma_cutoff=None,
    out=sys.stdout):
  anom_diff_arrays = []
  for input_array in input_arrays:
    if (not is_experimental_anomalous_array(input_array)): continue
    if not d_min: d_min=input_array.d_min() #2014-01-02

    processed_diffs = anomalous_diffs.process(
      observations=input_array,
      d_max=d_max,
      d_min=d_min,
      rms_cutoff=rms_cutoff,
      out=out)
    if (processed_diffs is not None):
      anom_diff_arrays.append((input_array, processed_diffs))
  if (len(anom_diff_arrays) == 0):
    for input_array in input_arrays:
      if (not is_anomalous_difference_array(input_array)): continue
      if not d_min: d_min=input_array.d_min() #2014-01-02
      processed_diffs = resolution_cutoff.process(
        observations=input_array,
        d_max=d_max,
        d_min=d_min,
        sigma_cutoff=sigma_cutoff,
        out=out)
      if (processed_diffs is not None):
        anom_diff_arrays.append(processed_diffs)
    if len(anom_diff_arrays) == 0:
      raise Sorry("No anomalous data found!")
    print >>out,"Assuming data is pre-processed anomalous differences or FA structure factors."
    print >>out
    data = ( None, anom_diff_arrays[0] )
  else:
    data = anom_diff_arrays[0]
    for other_data in anom_diff_arrays[1:]:
      if (compare_anom_diff_arrays(data, other_data) > 0):
        data = other_data
    if (len(anom_diff_arrays) > 1):
      print >>out,"Selecting anomalous differences derived from this input array:"
      data[0].show_comprehensive_summary()
      print >>out
      print >>out,"The corresponding anomalous differences again:"
      data[1].set_info(
        "Strongest anomalous signal: %.3f" % data[1].info())
      data[1].show_comprehensive_summary()
      print >>out
  return data

def is_experimental_anomalous_array(input_array):
  if (not input_array.anomalous_flag()): return False
  if (not isinstance(input_array.data(), flex.double)): return False
  if (input_array.sigmas() is None): return False
  if (not isinstance(input_array.sigmas(), flex.double)): return False
  return True

def is_anomalous_difference_array(input_array):
  if (input_array.anomalous_flag()): return False
  if (not isinstance(input_array.data(), flex.double)): return False
  if (input_array.sigmas() is None): return False
  if (not isinstance(input_array.sigmas(), flex.double)): return False
  return True

def prompt_for_symmetry_if_necessary(command_line_symmetry, reflection_file,
    unit_cell=None, space_group_info=None,out=sys.stdout):


  have_data = False
  ambiguous_unit_cell = False
  ambiguous_space_group_info = False
  for input_array in reflection_file.as_miller_arrays():
    if (is_experimental_anomalous_array(input_array)): have_data = True
    if (is_anomalous_difference_array(input_array)): have_data = True
    if (input_array.unit_cell() is not None):
      if (unit_cell is None):
        unit_cell = input_array.unit_cell()
      elif (not unit_cell.is_similar_to(input_array.unit_cell())):
        ambiguous_unit_cell = True
    if (input_array.space_group_info() is not None):
      if (space_group_info is None):
        space_group_info = input_array.space_group_info()
      elif (space_group_info.group()!=input_array.space_group_info().group()):
        ambiguous_space_group_info = True
  if (not have_data):
    if unit_cell and space_group_info:  # take input unit cell and sg
      return crystal.symmetry(
        unit_cell=unit_cell,
        space_group_info=space_group_info)
    else:
      return command_line_symmetry

  if unit_cell is not None:
    pass # it is already set
  elif (  (command_line_symmetry is not None) and
      (command_line_symmetry.unit_cell() is not None)) :
    unit_cell = command_line_symmetry.unit_cell()
  else:
    status = None
    correct = ""
    if (unit_cell == None):
      status = "unknown"
    elif (ambiguous_unit_cell):
      status = "ambiguous"
      correct = " correct"
    if (status is not None):
      print >>out,"Unit cell is %s. Please enter the%s" % (status, correct)
      while 1:
        parameters = input_with_prompt("unit cell parameters: ").input
        if (len(parameters.strip()) == 0): continue
        try:
          unit_cell = uctbx.unit_cell(parameters)
          break
        except Exception, e:
          print >>out,e
  if space_group_info is not None:
    pass # already ok
  elif ((command_line_symmetry is not None) and
      (command_line_symmetry.space_group_info() is not None)) :
    space_group_info = command_line_symmetry.space_group_info()
  else:
    status = None
    correct = ""
    if (space_group_info == None):
      status = "unknown"
    elif (ambiguous_space_group_info):
      status = "ambiguous"
      correct = " correct"
    if (status is not None):
      print >>out,"Space group is %s. Please enter the%s" % (status, correct)
    elif (symbol_confidence.level(space_group_info) == 0):
      print >>out,"Space group found in file:", space_group_info
      while 1:
        response = input_with_prompt(
          "Is this the correct space group? [Y/N]: ").input
        response = response.strip()[:1].upper()
        if (response == "Y"):
          break
        if (response == "N"):
          status = "unknown"
          print >>out,"Please enter the correct"
          break
    if (status is not None):
      while 1:
        symbol = input_with_prompt("space group symbol: ").input
        if (len(symbol.strip()) == 0): continue
        try:
          space_group_info = sgtbx.space_group_info(symbol=symbol)
          if (space_group_info.group().is_compatible_unit_cell(unit_cell)):
            break
          raise Sorry("Space group is incompatible with unit cell parameters.")
        except Exception, e:
          print >>out,e
  return crystal.symmetry(
    unit_cell=unit_cell,
    space_group_info=space_group_info)

def solve_interface(reflection_file_name, n_sites, scattering_type,
                    crystal_symmetry=None, d_max=None, d_min=None,
                    enable_early_termination=True,out=sys.stdout):
  reflection_file = reflection_file_reader.any_reflection_file(
    reflection_file_name)
  input_arrays = reflection_file.as_miller_arrays(
    crystal_symmetry=crystal_symmetry,
    force_symmetry=True)
  anom_diffs = select_array_to_use(
    input_arrays=input_arrays,
    d_max=d_max,
    d_min=d_min,out=out)

  scatterer, general_positions_only = process_scattering_type(scattering_type)

  print >>out,"Starting the search procedure:"
  return hyss.search(
    f_obs=anom_diffs,
    substructure_params=hyss.substructure_parameters(
      n_sites=n_sites,
      min_distance=3.5,
      general_positions_only=general_positions_only),
    search_params=hyss.search_parameters(
      enable_early_termination=enable_early_termination))

def validate_params_basic (params,input_arrays=None) :
  if (input_arrays is None and params.data is None ) :
    raise Sorry("Reflections file must be defined.")
  elif (input_arrays is None and not os.path.isfile(params.data)) :
    raise Sorry("The file '%s' does not exist or is not a file." %
      params.data)
  elif (params.n_sites is None) or (params.n_sites < 1) :
    raise Sorry("Please define the number of heavy-atom sites.")
  elif (params.scattering_type is None) :
    raise Sorry("Please specify an element type (such as Se, Pt, Hg, etc.).")
  return True

###---------------------------------------------------------------------
### PHENIX GUI stuff
def validate_params (params,input_arrays=None) :
  validate_params_basic(params,input_arrays)
  if (params.space_group is None) :
    raise Sorry("Space group required!")
  elif (params.unit_cell is None) :
    raise Sorry("Unit cell required!")
  return True

class launcher (runtime_utils.target_with_save_result) :
  def run (self) :
    os.mkdir(self.output_dir)
    os.chdir(self.output_dir)
    result = run(args=self.args, out=sys.stdout)
    return result

def finish_job (result) :
  output_files = []
  stats = []
  if (result is None) : return ([], [])
  if (result.pdb_file is not None) and (os.path.isfile(result.pdb_file)) :
    output_files.append((result.pdb_file, "Heavy-atom sites"))
  if (result.xyz_file is not None) and (os.path.isfile(result.xyz_file)) :
    output_files.append((result.xyz_file, "Fractional coordinates"))
  stats.append(("Number of sites", result.n_sites))
  stats.append(("CC", format_value("%.3f", result.cc)))
  return (output_files, stats)

if (__name__ == "__main__"):
  run(sys.argv[1:])
